/'                            Slider (Trackbar)
The slider consists of a linear scale with an arrow that can be dragged with a
mouse to set a value within a certain range of integers. When the slider has 
keyboard focus, the left and right arrow keys (for vertical sliders, the up and 
down arrows) increase and decrease the value of the slider by one. The PageUp and
PageDown keys increase or decrease the value by one page unit, which is the separation 
between tick marks.

Optionally, the slider can have a pair of labels associated with it whose only 
function is to display the limits set into the control.

Since WinFBE does not (currently) provide a slider, this class will create one on
a WinFBE form and give properties and methods to use it. The class does not provide 
an event routine for when the slider value changes, but you can add one by following
the example in Step 6 below.

Usage example:
	1. main .BAS file
			#include "slider.inc"			'That's this file. Insert any necessary path information.

   2. Design time:
      Put a frame on the form to mark the location and size of the slider. If you're
      going to be using the optional buddy labels, add them to the form also. The
      location of the buddies doesn't matter, as they will get moved to the sides
      of the slider when it is initialized. Make the buddies high enough and wide
      enough to hold the text of the limiting values of the slider. The alignment 
      properties of the buddy labels will be set to make them look good on the slider.
      
      If you make the frame higher than wide, the slider will operate vertically.
      Because of a Windows quirk, vertical sliders will look a little bit different 
      from horizontal ones.
      
   3. Form file, module level:
         Dim Shared as Slider slide1

   4. Form Load event:
         slide1.Initialize(frmmain, frmmain.frame1)   'Creates the slider as a child of frmmain
                                                      'in the location and size of frame1
         slide1.SetRange(lowval, highval)             'default values 0 to 100
         slide1.SetBuddyLow( frmmain.Label_low)       'the low-range label (optional)
         slide1.SetBuddyHigh(frmmain.label_high)      'the high-range label (optional)
         slide1.SetVal(startval)                      'starting value of the slider

   5. When you need the value of the slider:
         number = slide1.GetVal()
      To set the value of the slider from the program
         slide1.SetVal(newvalue)
      Change the range of the slider (the buddy labels, if any, will automatically show the new limits)
         slide1.SetRange(newlow, newhigh)       'if you enter them in the wrong order, they will be swapped
      Retrieve the current range limits:
         slide1.GetRange(curlow, curhigh)

   6. If you need an event when the value changes, then check the Form/AllEvents box
      Every time the value of the slider is changed (and sometimes when it isn't), an event
      will happen with e.message = WM_SlideChange (=&h138) and e.lparam = slide.hslider. For example:
            Function frmMain_AllEvents( ByRef sender As wfxForm, ByRef e As EventArgs ) As LRESULT
               if e.lParam=slide.hslider andalso e.Message=&h138 then
                  print "Value = "; slide.getval()
               end if
               Function = 0
            End Function
      According to Wine.org, the message &h138 is WM_CTLCOLORSTATIC, which sets the 
      color of static controls (frames, labels, etc). Go figure.
'/

const as long WM_SlideChange=&h138

type Slider			'Also known as a trackbar
	as wfxForm form		            'the form that is the home of this slider
	as hwnd hslider		            'the handle of the the new slider
   as wfxLabel ptr lblLow=null      'pointer to the low-limit label
   as wfxLabel ptr lblHigh=null     'pointer to the high-limit label
   as boolean vert                  'oriented vertically
   as long high, low                'saving this because of backwards vertical controls
	declare sub Initialize(form as wfxform, frame as wfxframe)
	declare function GetVal() as long
	declare sub SetVal(newval as long)
	declare sub SetRange(RangeLow as long, RangeHigh as long)
	declare sub GetRange(byref RangeLow as long, byref RangeHigh as long)
   declare sub SetBuddyLow(hlow as wfxLabel)
   declare sub SetBuddyHigh(hhigh as wfxLabel)
end type

'There are other properties we could add, but for simplicity, I used my favorite defaults,
'including 10 tickmarks along the scale.

sub Slider.Initialize(form as wfxform, frame as wfxframe)
	dim as long idc = form.getnextctrlid()
	dim as long dstyle
	if frame.height < frame.width then
		dstyle = WS_VISIBLE or ws_border OR WS_TABSTOP OR TBS_AUTOTICKS OR TBS_HORZ OR TBS_TOP OR TBS_TOOLTIPS
      this.vert = false
	else
		dstyle = WS_VISIBLE or ws_border OR WS_TABSTOP OR TBS_AUTOTICKS OR TBS_VERT OR TBS_right ' OR TBS_TOOLTIPS
      'On vertical sliders, the tooltips would be misleading
      this.vert = true
	end if
	
	
   this.hslider = form.pwindow->AddControl(	"TrackBar", , idc, "", _
															frame.left, frame.top, _
															frame.width, frame.height, _
															dstyle)
   frame.Visible = false	
end sub

/'	From WinFBX:
		DECLARE FUNCTION AddControl(  BYREF wszClassName AS WSTRING, _
												BYVAL hParent AS HWND = NULL, _
												BYVAL cID AS LONG_PTR = 0, _
                                    BYREF wszTitle AS WSTRING = "", _
												BYVAL x AS LONG = 0, BYVAL y AS LONG = 0, _
												BYVAL nWidth AS LONG = 0, BYVAL nHeight AS LONG = 0, _ 
												BYVAL dwStyle AS LONG = -1, BYVAL dwExStyle AS LONG = -1, _
												BYVAL lpParam AS LONG_PTR = 0, _
                                    BYVAL pWndProc AS SUBCLASSPROC = NULL, _
												BYVAL uIdSubclass AS UINT_PTR = &HFFFFFFFF, _
												BYVAL dwRefData AS DWORD_PTR = NULL) _
      AS HWND
'/

function slider.GetVal() as long
   if this.vert then
      return this.high + this.low - trackbar_getpos(this.hslider)
   else
      return trackbar_getpos( this.hslider )
   end if
end function

sub slider.SetVal( newval as long )
   if this.vert then
      trackbar_setpos(this.hslider, true, this.high+this.low-newval)
   else
      trackbar_setpos( this.hslider, true, newval)
   end if
end sub

sub slider.SetRange(RangeLow as long, RangeHigh as long)
   if (rangehigh < rangelow) then 
      swap rangelow, rangehigh 
   end if
   this.low = rangelow
   this.high = rangehigh
	trackbar_setrangemin( this.hslider, rangelow, true)
	trackbar_setrangemax( this.hslider, rangehigh, true)
   trackbar_setticfreq(this.hslider, clng((rangehigh-rangelow)/10))
   trackbar_setpagesize(this.hslider, clng((rangehigh-rangelow)/10))
   if this.vert then
      if this.lblLow <> null  then 
         this.lblLow->Text  = str(rangehigh)
         this.lblLow->TextAlign = LabelAlignment.BottomCenter
      end if
      if this.lblHigh <> null then 
         this.lblHigh->Text = str(rangelow)
         this.lblHigh->TextAlign = LabelAlignment.TopCenter
      end if
   else
      if this.lblLow <> null  then 
         this.lblLow->Text  = str(rangelow)
         this.lblLow->TextAlign = LabelAlignment.MiddleRight
      endif
      if this.lblHigh <> null then 
         this.lblHigh->Text = str(rangehigh)
         this.lblHigh->TextAlign = LabelAlignment.MiddleLeft   
      endif  
   end if
end sub

sub slider.GetRange(byref RangeLow as long, byref RangeHigh as long)
	rangelow = trackbar_getrangemin(this.hslider)
	rangehigh = trackbar_getrangemax(this.hslider)
end sub

sub slider.SetBuddyLow(hlow as wfxLabel)
   trackbar_setbuddy(this.hslider, true, hlow.hwindow)
   this.lblLow = @hlow
end sub

sub slider.SetBuddyHigh(hhigh as wfxLabel)
   trackbar_setbuddy(this.hslider, false, hhigh.hWindow)
   this.lblHigh = @hhigh
end sub



